/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_data_adaptor.cpp
 *
 * @brief [LEVEL: beta] implementation of @ref
 * diagnostics::unittest::Stream_Test_Data_Adaptor class
 *
 * $Id: stream_test_data_adaptor.cpp 861 2009-04-18 20:14:51Z tautschnig $
 * 
 * @author Christian Schallhart
 */

#include <diagnostics/unittest/test_system/stream_test_data_adaptor.hpp>

#include <diagnostics/unittest/test_exception.hpp>
#include <diagnostics/util/diff.hpp>
#include <diagnostics/util/diff_exception.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;

Stream_Test_Data_Adaptor::~Stream_Test_Data_Adaptor()
{
}

Stream_Test_Data_Adaptor::Stream_Test_Data_Adaptor(::std::istream & input,
						   ::std::ostream & output) 
    : m_input(input),
      m_output(output),
      m_mode(MODE_INTERACTIVE)
{
}

Stream_Test_Data_Adaptor::Mode_t Stream_Test_Data_Adaptor::mode() const
{
    return m_mode;
}

void Stream_Test_Data_Adaptor::mode(Mode_t const mode)
{
    m_mode=mode;
}


bool Stream_Test_Data_Adaptor::p_missing_test_data(::std::string const & id, 
						   ::std::string & data) const
{
    if(m_mode==MODE_ACCEPT) return true;
    if(m_mode==MODE_REJECT) return false;
		
    m_output << "MISSING ENTRY: ID='" << id << "'\n"
	     << "--------------------------------------------------------------------------------\n"
	     << data << "\n"
	     << "--------------------------------------------------------------------------------"
	     << ::std::endl;
	
    m_output << "Possible entries:\n" 
	     << " 'a': Accept the difference and save.\n"
	     << " 'r': Reject the difference and do not save.\n"
	     << "Proceed? [a|r]: ";

    char c_prev, c = '\n';
    bool first_run = true;
    for(;;){
	c_prev = c;
	m_input.get(c);
		
	// Remove all the newline trash that may stick in the input stream
	if(first_run && c == '\n'){
	    m_input.get(c);
	    first_run = false;
	}else if(first_run){
	    first_run = false;
	}
		
	if(m_input.eof())
	    break;

	if((c_prev=='\n') && (c=='A' || c=='a')){
	    if(m_input.get() == '\n'){
		m_output << "Accepting. As soon as you quit data will be saved to the database." << ::std::endl;
		return true;
	    }
	}else if((c_prev=='\n') && (c=='R' || c=='r')){
	    if(m_input.get() == '\n'){
		m_output << "Rejecting. Data will not be saved to the database." << ::std::endl;
		return false;
	    }
	}else if(c=='\n'){
	    m_output << "Possible entries:\n" 
		     << " 'a': Accept the difference and save.\n"
		     << " 'r': Reject the difference and do not save.\n"
		     << "Proceed? [a|r]: ";
	}
    }

    m_output << "This must not happen!" << ::std::endl;
    return false;
}
	
bool Stream_Test_Data_Adaptor::p_mismatching_test_data(::std::string const & id,
						       ::std::string const & data,
						       ::std::string & new_data) const
{
    using ::diagnostics::internal::diff_text;

    if(m_mode==MODE_ACCEPT) return true;
    if(m_mode==MODE_REJECT) return false;
	    
    char const * proceed_text( 
            "Possible entries:\n" 
		    " 'a': Accept the difference and save.\n"
		    " 'r': Reject the difference and do not save.\n"
		    " 'o': Print the old output.\n"
		    " 'n': Print the new output.\n"
		    " 'd': Print the diff between the old and new output\n"
		    "Proceed? [a|r|o|n|d]: " );

    try
    {
    m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
	     << "Diff:\n"
	     << "--------------------------------------------------------------------------------\n"
	     << diff_text(data, new_data, true) << "\n"
	     << "--------------------------------------------------------------------------------"
	     << ::std::endl;
    
    m_output << proceed_text;
    }
    catch(::diagnostics::internal::Diff_Exception const& e)
    {
    m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
         << e.what() << ::std::endl;
    }
    
    char c_prev, c = '\n';
    bool first_run = true;
    for(;;){
	c_prev = c;
	m_input.get(c);
	
	// Remove all the newline trash that may stick in the input stream
	if(first_run && c == '\n'){
	    m_input.get(c);
	    first_run = false;
	}else if(first_run){
	    first_run = false;
	}
		
	if(m_input.eof())
	    break;

	if((c_prev=='\n') && (c=='A' || c=='a')){
	    if(m_input.get() == '\n'){
		m_output << "Accepting. As soon as you quit data will be saved to the database." << ::std::endl;
		return true;
	    }
	}else if((c_prev=='\n') && (c=='R' || c=='r')){
	    if(m_input.get() == '\n'){
		m_output << "Rejecting. Data will not be saved to the database." << ::std::endl;
		return false;
	    }
	}else if((c_prev=='\n') && (c=='O' || c=='o')){
	    c_prev = c;
	    m_input.get(c);
	    if(c == '\n'){
		m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
			 << "Old output:\n"
			 << "--------------------------------------------------------------------------------\n"
			 << data << "\n"
			 << "--------------------------------------------------------------------------------"
			 << ::std::endl;
		    	
		m_output << proceed_text;
	    }
	}else if((c_prev=='\n') && (c=='N' || c=='n')){
	    c_prev = c;
	    m_input.get(c);
	    if(c == '\n'){
		m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
			 << "New output:\n"
			 << "--------------------------------------------------------------------------------\n"
			 << new_data << "\n"
			 << "--------------------------------------------------------------------------------"
			 << ::std::endl;
		    	
		m_output << proceed_text;
	    }
	}else if((c_prev=='\n') && (c=='D' || c=='d')){
	    c_prev = c;
	    m_input.get(c);
	    if(c == '\n'){
        try
        {
		m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
			 << "Diff:\n"
			 << "--------------------------------------------------------------------------------\n"
			 << diff_text(data, new_data, true) << "\n"
			 << "--------------------------------------------------------------------------------"
			 << ::std::endl;
        }
        catch(::diagnostics::internal::Diff_Exception const& e)
        {
		m_output << "MISSMATCHING ENTRY: ID='" << id << "'\n"
             << e.what() << ::std::endl;
        }
		    	
		m_output << proceed_text;
	    }
	}else if(c=='\n'){
	    m_output << proceed_text;
	}
    }

    m_output << "This must not happen!" << ::std::endl;
    return false;
}
	
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;


// vim:ts=4:sw=4
