﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.CommandLine;
using System.CommandLine.Builder;
using System.CommandLine.Invocation;
using System.CommandLine.Parsing;
using System.IO;
using System.Threading;
using System.Threading.Tasks;

using DiagnosticsReleaseTool.Impl;

namespace DiagnosticsReleaseTool.CommandLine
{
    internal sealed class DiagnosticsReleaseCommandLine
    {
        private static async Task<int> Main(string[] args)
        {
            Parser parser = new CommandLineBuilder()
                .AddCommand(PrepareRelease())
                .CancelOnProcessTermination()
                .UseDefaults()
                .Build();

            return await parser.InvokeAsync(args).ConfigureAwait(false);
        }

        public static Command PrepareRelease() =>
            new(
                name: "prepare-release",
                description: "Given a darc drop, generates validated manifests and layouts to initiate a tool release.")
            {
                CommandHandler.Create<Config, bool, CancellationToken>(DiagnosticsReleaseRunner.PrepareRelease),
                // Inputs
                InputDropPathOption(), ToolManifestPathOption(), ReleaseNameOption(),
                // Toggles
                ToolManifestVerificationOption(), DiagnosticLoggingOption(),
                // Outputs
                StagingPathOption(),
                AzureStorageAccountNameOption(), AzureStorageAccountKeyOption(), AzureStorageContainerNameOption()
            };


        private static Option<bool> DiagnosticLoggingOption() =>
            new(
                aliases: ["-v", "--verbose"],
                description: "Enables diagnostic logging",
                getDefaultValue: () => false);

        private static Option ToolManifestPathOption() =>
            new Option<FileInfo>(
                aliases: ["--tool-manifest", "-t"],
                description: "Full path to the manifest of tools and packages to publish.")
            {
                IsRequired = true
            }.ExistingOnly();

        private static Option<bool> ToolManifestVerificationOption() =>
            new(
                alias: "--verify-tool-manifest",
                description: "Verifies that the assets being published match the manifest",
                getDefaultValue: () => true);

        private static Option<DirectoryInfo> InputDropPathOption() =>
            new Option<DirectoryInfo>(
                aliases: ["-i", "--input-drop-path"],
                description: "Path to drop generated by `darc gather-drop`")
            {
                IsRequired = true
            }.ExistingOnly();

        private static Option<string> ReleaseNameOption() =>
            new(
                aliases: ["-r", "--release-name"],
                description: "Name of this release.")
            {
                IsRequired = true,
            };

        private static Option StagingPathOption() =>
            new Option<DirectoryInfo>(
                aliases: ["--staging-directory", "-s"],
                description: "Full path to the staging path.",
                getDefaultValue: () => new DirectoryInfo(
                    Path.Join(Path.GetTempPath(), Path.GetRandomFileName())))
            .LegalFilePathsOnly();

        private static Option<string> AzureStorageAccountNameOption() =>
            new(
                aliases: ["-n", "--account-name"],
                description: "Storage account name, must be in public azure cloud.")
            {
                IsRequired = true,
            };

        private static Option<string> AzureStorageAccountKeyOption() =>
            new(
                aliases: ["-k", "--client-id"],
                description: "Identity Client ID. If left blank, ambient identity will be used.",
                getDefaultValue: () => null)
            {
                IsRequired = true,
            };

        private static Option<string> AzureStorageContainerNameOption() =>
            new(
                aliases: ["-c", "--container-name"],
                description: "Storage account container name where the files will be uploaded.")
            {
                IsRequired = true,
            };
    }
}
