/* $Id: glue-audio.c,v 1.26 2009-04-24 20:07:44 potyra Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <sys/time.h>
#include <assert.h>
#include <fcntl.h>
#include <inttypes.h>
#include <setjmp.h>
#include <signal.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#include "config.h"

#include "glue-audio_driver.h"
#include "glue-log.h"
#include "glue-main.h"

#include "sig_sound.h"	/* FIXME */

static const char *desired_audio_module = NULL;


/* currently implemented drivers */

#ifdef USE_OSS_AUDIO
extern ao_functions_t audio_out_oss;
#endif
#ifdef USE_ESD
extern ao_functions_t audio_out_esd;
#endif
#ifdef USE_PULSE
extern ao_functions_t audio_out_pulse;
#endif
#ifdef HAVE_ALSA5
extern ao_functions_t audio_out_alsa5;
#endif
#ifdef HAVE_ALSA9
extern ao_functions_t audio_out_alsa9;
#endif
#ifdef HAVE_ALSA1X
extern ao_functions_t audio_out_alsa1x;
#endif
#ifdef USE_MACOSX_AUDIO
extern ao_functions_t audio_out_macosx;
#endif
extern ao_functions_t audio_out_null;
extern ao_functions_t audio_out_file;

ao_functions_t* audio_out_drivers[] =
{
// wrappers:
#ifdef USE_PULSE
	&audio_out_pulse,
#endif
#ifdef USE_ESD
	&audio_out_esd,
#endif

// native:
#ifdef USE_MACOSX_AUDIO
	&audio_out_macosx,
#endif
#ifdef HAVE_ALSA1X
	&audio_out_alsa1x,
#endif
#ifdef HAVE_ALSA9
	&audio_out_alsa9,
#endif
#ifdef HAVE_ALSA5
	&audio_out_alsa5,
#endif
#ifdef USE_OSS_AUDIO
        &audio_out_oss,
#endif
        &audio_out_null,
	&audio_out_file,
	NULL
};

ao_functions_t *audio_out = NULL;

static enum { STALLED, PLAYING } state;
static uint8_t buffer[0x40000];
static unsigned int bufsize;
static unsigned long long next_event;


static void
audio_flush(void)
{
	unsigned int count;
	int ret;

	switch (state) {
	case STALLED:
		if (bufsize < sizeof(buffer) / 2) {
			return;
		}
		// fprintf(stderr, "%s: started\n", __FUNCTION__);
		state = PLAYING;
		break;

	case PLAYING:
		if (audio_out->get_delay() == 0.0) {
			// fprintf(stderr, "%s: stopped\n", __FUNCTION__);
			state = STALLED;
			return;
		}
		break;

	default:
		assert(0);
	}

	count = audio_out->get_space();
	if (bufsize < count) {
		count = bufsize;
	}
	if(count > 0) {
		ret = audio_out->play(buffer, count);
		assert(0 <= ret && ret <= count);

		memmove(buffer, &buffer[ret], bufsize - ret);
		bufsize -= ret;
	}
}

void
audio_play(int16_t *buf)
{
	if (! audio_out) {
		/* We don't have any audio output device. */
		return;
	}

	if (bufsize + SIG_SOUND_MTU < sizeof(buffer)) {
		memcpy(&buffer[bufsize], buf, SIG_SOUND_MTU);
		bufsize += SIG_SOUND_MTU;
	} else {
		/* Skip bytes! */
		// fprintf(stderr, "%s: skipping frame.\n", __FUNCTION__);
	}

	audio_flush();
}

extern void
audio_event(void *s)
{
	audio_flush();

	next_event += TIME_HZ / 128;
	time_call_at(next_event, audio_event, (void *) 0);
}

void
audio_init(void)
{
	if (audio_out) {
		next_event = 0;
		time_call_at(next_event, audio_event, (void *) 0);
	}
}

void
audio_exit(void)
{
}

static sigjmp_buf audio_create_state;

static void __attribute__((__noreturn__))
audio_create_break(int sig)
{
	audio_out = NULL;

	siglongjmp(audio_create_state, 1);
}

void
audio_create(void)
{
	struct sigaction sa, osa;
	struct itimerval it, oit;
	int i,ret;

	/*
	 * Abort audio initialization after about 1sec.
	 */
	if (sigsetjmp(audio_create_state, 1) != 0) {
		goto abort;
	}

	/*
	 * Set alarm signal handler and real-time timer.
	 */
	sa.sa_handler = audio_create_break;
	sa.sa_flags = 0;
	sigfillset(&sa.sa_mask);
	ret = sigaction(SIGALRM, &sa, &osa);
	assert(0 <= ret);

	it.it_interval.tv_sec = 0;
	it.it_interval.tv_usec = 0;
	it.it_value.tv_sec = 5; 
	it.it_value.tv_usec = 0;
	ret = setitimer(ITIMER_REAL, &it, &oit);
	assert(0 <= ret);

	/* Try opening audio device. */
	for (i=0; audio_out_drivers[i]; i++){
		if ((desired_audio_module == NULL
		     || strcmp(audio_out_drivers[i]->info->short_name,
			       desired_audio_module) == 0)
		    && audio_out_drivers[i]->init()) {
			audio_out = audio_out_drivers[i];
			break;
		}
	}

abort:	;
	/*
	 * Restore real-time timer and alarm signal handler.
	 */
	ret = setitimer(ITIMER_REAL, &oit, (struct itimerval *) 0);
	assert(0 <= ret);

	ret = sigaction(SIGALRM, &osa, (struct sigaction *) 0);
	assert(0 <= ret);

	if (audio_out) {
		faum_log(FAUM_LOG_INFO, progname, "",
				"Using %s.\n", audio_out->info->name);
	} else {
		faum_log(FAUM_LOG_ERROR, progname, "",
				"Can't open any audio output device.\n");
		return;
	}

	state = STALLED;
	bufsize = 0;
}

void
audio_destroy(void)
{
	if(audio_out) {
		audio_out->uninit(0);
		audio_out = NULL;
	}
}

void
audio_usage(void)
{
	fprintf(stderr, "\t-a <audio_module>: use <audio_module> during "
			"simulation\n");
}

int
audio_handle_args(int *argc, char **argv)
{
	int na = *argc;

	desired_audio_module = getenv("FAUM_AUDIO");

	while (na > 0) {
		assert(argv != NULL);
		if (*argv && strcmp(*argv, "-a") == 0) {
			(*argc)--;
			na--;

			if (*(argv + 1) == NULL) {
				return -1;
			}

			desired_audio_module = *(argv + 1);
			(*argc)--;
			na--;

			/* remove these two arguments. */
			memmove(argv, argv + 2, 
				(na + 1)  * sizeof(char *));
			continue;
		}

		na--; argv++;
	}

	return 0;
}
