/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_config.h>
#include <globus_io.h>
#include <globus_usage.h>
#include "globus_rls_client.h"
#include "version.h"
#include "misc.h"
#include "usage.h"
#include <stdarg.h>
#include <termios.h>
#include <string.h>

extern int	loglevel;

#define USAGE_BUFSIZE       256
#define USAGE_CODE          7
#define USAGE_VER           0
#define USAGE_COUNT         13
#define USAGE_FIELD_VER     "VER"
#define USAGE_FIELD_UPTIME  "UPTIME"
#define USAGE_FIELD_LRC     "LRC"
#define USAGE_FIELD_RLI     "RLI"
#define USAGE_FIELD_LFN     "LFN"
#define USAGE_FIELD_PFN     "PFN"
#define USAGE_FIELD_MAP     "MAP"
#define USAGE_FIELD_RLILFN  "RLILFN"
#define USAGE_FIELD_RLILRC  "RLILRC"
#define USAGE_FIELD_RLISND  "RLISND"
#define USAGE_FIELD_RLIMAP  "RLIMAP"
#define USAGE_FIELD_THRD    "THRD"
#define USAGE_FIELD_CONN    "CONN"
#define USAGE_TRUE          1
#define USAGE_FALSE         0

static globus_usage_stats_handle_t  usage_handle = NULL;

/*
 * Initialize the globus_usage handle for subsequent use. No stats should be
 * send before this function is called.
 */
void
usage_init()

{
  globus_result_t   r;
  char              errmsg[MAXERRMSG];

  if (loglevel >= 3)
    logit(LOG_DEBUG, "usage_init:");

  if ((r = globus_usage_stats_handle_init(&usage_handle, USAGE_CODE,
        USAGE_VER, NULL)) != GLOBUS_SUCCESS) {
    logit(LOG_WARNING, "usage_init: globus_usage_stats_handle_init: %s",
    globuserr(r, errmsg));
  }
}

/*
 * Destroy the globus_usage handle. No stats should be sent after this
 * function is called.
 */
void
usage_destroy()

{
  if (loglevel >= 3)
    logit(LOG_DEBUG, "usage_destroy:");

  if(usage_handle != NULL)
  {
    globus_usage_stats_handle_destroy(usage_handle);
  }
}

/*
 * Formats the stats output and calls the globus_usage send function to
 * send stats to the designated stats reciever(s).
 */
void
usage_send(
    int             lrc,
    int             rli,
    int             uptime,
    int             numlfn,
    int             numpfn,
    int             nummap,
    int             numrlilfn,
    int             numrlilrc,
    int             numrlisender,
    int             numrlimap,
    int             threadcount,
    int             conncount)

{
    char            ver_b[USAGE_BUFSIZE];
    char            uptime_b[USAGE_BUFSIZE];
    char            lrc_b[USAGE_BUFSIZE];
    char            rli_b[USAGE_BUFSIZE];
    char            numlfn_b[USAGE_BUFSIZE];
    char            numpfn_b[USAGE_BUFSIZE];
    char            nummap_b[USAGE_BUFSIZE];
    char            numrlilfn_b[USAGE_BUFSIZE];
    char            numrlilrc_b[USAGE_BUFSIZE];
    char            numrlisender_b[USAGE_BUFSIZE];
    char            numrlimap_b[USAGE_BUFSIZE];
    char            threadcount_b[USAGE_BUFSIZE];
    char            conncount_b[USAGE_BUFSIZE];

    globus_result_t r;
    char            errmsg[MAXERRMSG];

    if (loglevel >= 3)
      logit(LOG_DEBUG, "usage_send:");

    sprintf(ver_b, "%d.%d", MAJOR, MINOR);
    sprintf(uptime_b, "%d", uptime);
    sprintf(lrc_b, "%d", (lrc) ? USAGE_TRUE : USAGE_FALSE);
    sprintf(rli_b, "%d", (rli) ? USAGE_TRUE : USAGE_FALSE);
    sprintf(numlfn_b, "%d", numlfn);
    sprintf(numpfn_b, "%d", numpfn);
    sprintf(nummap_b, "%d", nummap);
    sprintf(numrlilfn_b, "%d", numrlilfn);
    sprintf(numrlilrc_b, "%d", numrlilrc);
    sprintf(numrlisender_b, "%d", numrlisender);
    sprintf(numrlimap_b, "%d", numrlimap);
    sprintf(threadcount_b, "%d", threadcount);
    sprintf(conncount_b, "%d", conncount);

    r = globus_usage_stats_send(
            usage_handle,
            USAGE_COUNT,
            USAGE_FIELD_VER,    ver_b,
            USAGE_FIELD_UPTIME, uptime_b,
            USAGE_FIELD_LRC,    lrc_b,
            USAGE_FIELD_RLI,    rli_b,
            USAGE_FIELD_LFN,    numlfn_b,
            USAGE_FIELD_PFN,    numpfn_b,
            USAGE_FIELD_MAP,    nummap_b,
            USAGE_FIELD_RLILFN, numrlilfn_b,
            USAGE_FIELD_RLILRC, numrlilrc_b,
            USAGE_FIELD_RLISND, numrlisender_b,
            USAGE_FIELD_RLIMAP, numrlimap_b,
            USAGE_FIELD_THRD,   threadcount_b,
            USAGE_FIELD_CONN,   conncount_b);

    if (r != GLOBUS_SUCCESS) {
        logit(LOG_WARNING, "usage_send: globus_usage_stats_send: %s",
        globuserr(r, errmsg));
    }
}
