/*
Copyright (C) 2000-2013  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.*/

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "header.h"
#include "patchfunc.h"
extern int indent;

int has_iter(char *s)
{
  int nf=getfunc(s);
  return !!lfunc[nf].iter;
}

void patchfunction(struct patchfunc_s *p)
{
  int nf=getfunc(p->gpname);
  lfunc[nf].proto.code=p->code;
  functype(lfunc[nf])=p->type;
  funcmode(lfunc[nf])=p->mode;
  lfunc[nf].spec=p->spec;
}

void patchfunclist(void)
{
  int i;
  for(i=0;patchfunc[i].gpname;i++)
    patchfunction(patchfunc+i);
  if (!has_iter("forvec"))
  {
    struct patchfunc_s pforvec = {"forvec",NULL,Gnotype,(1<<Msemicolon),GPforvec};
    patchfunction(&pforvec);
  }
  if (!has_iter("forprime"))
  {
    struct patchfunc_s pforprime = {"forprime",NULL,Gnotype,(1<<Msemicolon),GPforprime};
    patchfunction(&pforprime);
  }
}
void printlistfunc(FILE *fout)
{
  int i;
  for(i=0;lfunc[i].gpname;i++)
    fprintf(fout,"%s %s %s\n",lfunc[i].gpname,lfunc[i].proto.cname,(lfunc[i].proto.code?lfunc[i].proto.code:""));
}
void checkisvar(int aff, const char *func, int *var, int *t)
{
  *var=aff;
  switch(tree[*var].f)
  {
    case Fentry:
      *t=Gnotype;
      break;
    case Ftag:
      *t=tree[*var].y;
      *var=tree[*var].x;
      if (tree[*var].f!=Fentry)
        die(aff,"incorrect syntax for %s",func);
      break;
    default:
      die(aff,"incorrect syntax for %s",func);
  }
}

int guesstype(int n)
{
  switch(tree[n].f)
  {
  case Fsmall:
    return Gsmall;
  case Fnoarg:
    return Gvoid;
  case Ftag:
    return tree[n].y;
  case Fmat: case Fvec:
    return Gvec;
  case Fconst:
    switch(value[tree[n].x].type)
    {
    case CSTsmall:
      return Gsmall;
    case CSTint:
      return Gint;
    case CSTsmallreal:
    case CSTreal:
      return Greal;
    case CSTstr:
      return Gstr;
    }
  case Fentry:
    {
      long v = getvar(n);
      return v<0 ? Ggen: vartype(ctxstack[v]);
    }
  case Ffunction:
    if (!strcmp(entryname(n),"length"))
      return Glg;
  default: /*FALL THROUGH*/
    return Ggen;
  }
}

void genequal(int aff, const char *func, int *var, int *binf, int *t)
{
  if (tree[aff].f!=Fassign)
    die(aff,"incorrect syntax for %s",func);
  checkisvar(tree[aff].x,func,var,t);
  *binf=tree[aff].y;
}
void genequaltosmall(int binf,int bsup, int *tv, int def)
{
  if (*tv==Gnotype)
  {
    int tinf=guesstype(binf);
    int tsup=guesstype(bsup);
    if  (is_subtype(tinf,Gsmall) && is_subtype(tsup,Gsmall))
      *tv=Gsmall;
    else
      *tv=def;
  }
}

int gengerepile(int bl)
{
  int arg[3];
  block[bl].gc|=(1<<GCneeded);
  arg[0]=newnode(Fentry,newentry("btop"),-1);
  pushvar(arg[0],0,Gpari_sp,newcall("_avma",-1));
  arg[1]=newsmall(bl);
  if (FC_gc_needed>=0)
    arg[2]=GNOARG;
  else
  {
    arg[2]=newnode(Fentry,newentry("st_lim"),-1);
    pushvar(arg[2],0,Gpari_sp,
        newcall("_stack_lim",newnode(Flistarg,arg[0],newsmall(1))));
  }
  return stacktoargs(arg,3);
}

int gengerepilereturn(int bl)
{
  int arg[2];
  arg[0]=newnode(Fentry,newentry("ltop"),-1);
  arg[1]=newsmall(bl);
  return stacktoargs(arg,2);
}

int newgetg(int v, int l, const char *t)
{
  return newcall("_cgetg", newnode(Flistarg, newnode(Flistarg,
          v, l), newstringnode(t,-1)));
}

void makesubblock_ctx(int n, int savx)
{
  int bl3=newblock();
  int level = preclevel;
  genblock(n,-1);
  makeblock(bl3,n,newleaf(n),-1,savx);
  preclevel = level;
}

void makesubblock(int n)
{
  makesubblock_ctx(n, s_ctx.n);
}

Clabel_t genblock_label(int n, int p, int type)
{
  Clabel_t t;
  static int labie=0;
  int l=newlabel(type);
  if (type==Liferr)
    label[l].ie=labie++;
  genblock(n,p);
  if (type==Liferr)
    labie--;
  t=label[l];
  stack_pop_safe(&s_label,l);
  return t;
}

Clabel_t makesubblock_label(int n, int t)
{
  int savx=s_ctx.n;
  int bl3=newblock();
  Clabel_t lab=genblock_label(n,-1,t);
  makeblock(bl3,n,newleaf(n),-1,savx);
  return lab;
}

int check_labels(int k, int t)
{
  int l=s_label.n;
  while(k--)
  {
    if (!l) return 2;
    if (label[--l].type==t)
      return 1;
  }
  return 0;
}

int get_iferr(int k)
{
  int i, l=s_label.n;
  for (i=s_label.n-k; i<l; i++)
  {
    if (label[i].type==Liferr)
      return label[i].ie;
  }
  return 0;
}

int
geniterproto(int n, long fl, gpfunc *gp, int *argo, int *seq, int *var, int *tv)
{
  int arg[STACKSZ];
  const char *p=gp->proto.code;
  char c;
  PPproto mod;
  int nb=genlistargs(n,arg,0,STACKSZ);
  int i=0, k=0, binf;
  while((mod=parseproto(&p,&c))!=PPend)
  {
    if (i<nb && tree[arg[i]].f!=Fnoarg && (mod==PPdefault || mod==PPdefaultmulti))
      mod=PPstd;
    switch(mod)
    {
    case PPstd:
      if (i>=nb)
        die(n,"too few arguments");
      switch(c)
      {
      case '=':
        genequal(arg[i], gp->gpname, var, &binf, tv);
        if (fl)
        {
          tree[*var].t = *tv;
          argo[k++] = *var;
          tree[binf].t = guesstype(binf);
        }
        argo[k++] = binf;
        break;
      case 'V':
        checkisvar(arg[i], gp->gpname, var, tv);
        if (fl)
        {
          tree[*var].t = *tv;
          argo[k++] = *var;
        }
        break;
      case 'I':
        *seq = arg[i];
        break;
      default:
        if (fl)
          tree[arg[i]].t = guesstype(arg[i]);
        argo[k++] = arg[i];
      }
      break;
    case PPdefault:
    case PPdefaultmulti:
      if (i < nb)
      {
        tree[arg[i]].t = guesstype(arg[i]);
        argo[k++] = arg[i];
      }
      break;
    case PPstar:
      i=nb-1;
      break;
    case PPauto: /* Skip */
      i--;
      break;
    default:
      break;
    }
    i++;
  }
  return k;
}

void
genblockfunciter(int n, gpfunc *gp)
{
  int arg[STACKSZ];
  int var=-1, tv=Gnotype;
  int bseq=-1,aseq=-1,ret=-1,vprime,seq,cseq;
  int gpm=-1,bl0,bl1,bl2;
  int sav0, savx;
  Clabel_t lab;
  gpiterator * iter;
  const char * init, *next;
  int nb = geniterproto(n, 1, gp, arg, &seq, &var, &tv);
  gpdescarg * da = descfindrulesdsc(nb, arg, gp->iter);
  if (!da) die(n, "no suitable iterator found");
  iter = &(da->iter);
  init = lfunc[iter->init].gpname;
  next = lfunc[iter->next].gpname;
  sav0=s_ctx.n;
  bl0=newblock();
  vprime=newnode(Fentry,newentry("iter"),-1);
  pushvar(vprime,1<<Cuser,iter->type,-1);
  nb = geniterproto(n, 0, gp, arg+1, &seq, &var, &tv)+1;
  arg[0] = vprime;
  bseq = newcall(init, stacktoargs(arg,nb));
  genblock(bseq,-1);
  savx=s_ctx.n;
  bl1=newblock();
  bl2=newblock();
  if (autogc)
    gpm=gengerepile(bl2);
  if (tv==Gnotype)
  {
    gpdescarg *dv;
    tree[vprime].t = iter->type;
    dv=descfindrules(1, &vprime, lfunc+iter->next);
    if (!dv) die(n, "no suitable iterator_next found");
    tv = dv->type;
  }
  pushvar(var,1<<Cuser|1<<Ccompo,tv,-1);
  lab=makesubblock_label(seq,Lbreak);
  if (lab.ne)
    seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
  if (autogc)
    seq=addseqright(seq,newcall("_gerepilemany",gpm));
  cseq=newnode1(Fassign,var,newcall(next,vprime));
  tree[cseq].m=Mparens;
  aseq=addseqright(aseq,newcall("_whilenull",newnode(Flistarg,cseq,seq)));
  if (lab.go)
    aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
  makeblocks3(bl0,bl1,bl2,n,bseq,aseq,ret,savx,savx);
  copyctx(sav0,block+bl0);
  restorectx(sav0);
}

/*
   The 6 operations which order is very important are
are:

  pushvar/newdecl(var)
  gpm=gengerepile(bl2);
  bseq=geninsertvar(arg,var)
  genblock(bseq,p)
  seq=addseqright(seq,newcall("_gerepilemany",gpm));
  savx=s_ctx.n

  savx=s_ctx.n start the recording of var in the block
  pushvar/newdecl(var) must be before geninsert(arg,var) obviously,

  if you use geninsertvar to store the value a a node, then
  geninsertvar must be called before genblock and genblock must be
  called on the geninsertvar node, with p=-1 since the return value is really
  ignored.
  G.N.^H^H^H^HB.A. (well it still lacks 20 lines more to really be G.N.)*/
void genblockfuncspec(int n, int p, gpfunc *gp)
{
  int arg[STACKSZ];
  int nb;
  int y=tree[n].y;
  int binf,bsup,seq;
  int var=-1,vinf,vsup,vprime;
  int bseq=-1,aseq=-1,ret=-1;
  int tv;
  int gpm=-1,bl1,bl2;
  int savx;
  Clabel_t lab;
  switch(gp->spec)
  {
  case GPaddhelp:
    {  /*prototype vSs*/
      gpfunc *gp;
      int d, nf;
      nb=genlistargs(n,arg,2,2);
      d=detag(arg[0]);
      if (tree[d].f!=Fentry && !is_const(d,CSTstr))
        die(n,"Incorrect function name in addhelp");
      tree[d].f=Fconst;
      nf=getfunc(entryname(d)); gp=lfunc+nf;
      d=detag(arg[1]);
      gp->proto.help=d;
    }
    break;
  case GPgrandO:
    if (tree[y].f==Ffunction && tree[y].x==OPpow)
      tree[y]=tree[tree[y].y];
    tree[n].x=newentry("O(_^_)");
    genblock(y,n);
    break;
  case GPif:
    nb=genlistargs(n,arg,2,STACKSZ);
    if (nb==3 && arg[2]==GNOARG)
      /*Some people type 'if(x!=0,print(x),)'*/
      nb--;
    if (nb==3 && arg[1]==GNOARG)
    {
      /*Some people type 'if(x!=0,,print(x))'*/
      arg[0]=newopcall(OPnb,arg[0],-1);
      arg[1]=arg[2];
      nb--;
    }
    if (tree[arg[0]].f==Fsmall && tree[arg[0]].x)
    {
      /*if(1,...) expr are often used to create block. We honor it here*/
      if (p>=0 && tree[p].f!=Fseq)
        newdecl((1<<Cauto)|(1<<Cconst),Gvoid,GNIL,&ret);
      arg[1]=geninsertvar(arg[1],ret);
      makesubblock(arg[1]);
      aseq=arg[1];
    }
    else
    {
      if (p>=0 && tree[p].f!=Fseq)
        newdecl((1<<Cauto)|(1<<Cimmutable),Gvoid,GNIL,&ret);
      if (nb>=4)
      {
        arg[2] = newcall("if", stacktoargs(arg+2,nb-2));
        nb = 3;
      }
      if (arg[0]>=0 && tree[arg[0]].f==Fseq)
      {
        int pred;
        newdecl((1<<Cauto)|(1<<Cimmutable),Gbool,GNIL,&pred);
        aseq = geninsertvar(arg[0], pred);
        genblock(aseq,n);
        arg[0] = newleaf(pred);
      }
      genblock(arg[0],n);
      if (arg[1]!=GNOARG)
      {
        arg[1]=geninsertvar(arg[1],ret);
        makesubblock(arg[1]);
      }
      if(nb==3)
      {
        arg[2]=geninsertvar(arg[2],ret==-1?-1:newleaf(ret));
        makesubblock(arg[2]);
      }
      aseq=addseqright(aseq,newnode(tree[n].f,tree[n].x,stacktoargs(arg,nb)));
    }
    makeblock(newblock(),n,aseq,ret,s_ctx.n);
    break;
  case GPinstall:
    {
      gpfunc *gp;
      const char *p;
      int d, nf;
      /* We should not use genblock(y,n) since prototype is
       * {rrD\"\",r,D\"\",s,}
       * Instead we fix Fentry to Fconst.
       */
      nb=genlistargs(n,arg,2,4);
      /*What to do with arg[3]? add a linking ??*/
      if (nb<3 || arg[2]==GNOARG) arg[2]=arg[0];
      d=detag(arg[2]);
      if (tree[d].f!=Fentry && !is_const(d,CSTstr))
        die(n,"Incorrect GP name in install");
      tree[d].f=Fconst;
      nf=getfunc(entryname(d)); gp=lfunc+nf;
      d=detag(arg[0]);
      if (tree[d].f!=Fentry && !is_const(d,CSTstr))
        die(n,"Incorrect C name in install");
      tree[d].f=Fconst;
      gp->proto.cname=entryname(d);
      if (nb==4 && arg[3]!=GNOARG)
      {
        d=detag(arg[3]);
        if (!is_const(d,CSTstr))
          die(n,"Incorrect library name in install");
        gp->proto.origin=entryname(d);
      }
      d=detag(arg[1]);
      if (tree[d].f!=Fentry && !is_const(d,CSTstr))
        die(n,"Incorrect code in install");
      tree[d].f=Fconst;
      p=entryname(d);
      gp->proto.code=p+1;
      funcmode(*gp)=0;
      switch(p[0])
      {
      case 'v':
        functype(*gp)=Gvoid;
        break;
      case 'l':
        functype(*gp)=Gsmall;
        break;
      case 'u':
        functype(*gp)=Gusmall;
        break;
      case 'i':
        functype(*gp)=Gsmall_int;
        break;
      case 'm':
        functype(*gp)=Ggen;
        funcmode(*gp)=(1<<Mcopy);
        break;
      default:
        functype(*gp)=Ggen;
        gp->proto.code--;
      }
      if (tree[arg[1]].f==Ftag)
        functype(*gp)=tree[arg[1]].y;
      for(;*p;p++) if (*p=='p' || *p=='b') funcmode(*gp)|=(1<<Mprec);
      gp->spec=GPinstalled;
    }
    break;
  case GPfor:
    nb=genlistargs(n,arg,3,3);
    genequal(arg[0],gp->gpname,&var,&binf,&tv);
    bsup=arg[1];seq=arg[2];
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vinf);
    bseq=addseqright(bseq,geninsertvar(binf,newleaf(vinf)));
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup);
    bseq=addseqright(bseq,geninsertvar(bsup,newleaf(vsup)));
    genblock(bseq,-1);
    savx=s_ctx.n;
    bl1=newblock();
    bl2=newblock();
    if (autogc)
      gpm=gengerepile(bl2);
    genequaltosmall(binf,bsup,&tv,Ggen);
    pushvar(var,1<<Cuser,tv,-1);
    lab=makesubblock_label(seq,Lbreak);
    if (lab.ne)
      seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
    if (autogc)
      seq=addseqright(seq,newcall("_gerepilemany",gpm));
    arg[0]=newnode1(Fassign,var,vinf);
    arg[1]=newopcall(OPle,newleaf(var),vsup);
    arg[2]=seq;
    arg[3]=newopcall(OPpp,newleaf(var),-1);
    aseq=newnode(tree[n].f,tree[n].x,stacktoargs(arg,4));
    if (lab.go)
      aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
    makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    break;
  case GPforvec:
    {
      int flag, vfvdat,vfvnext,next,sav0,bl0;
      nb=genlistargs(n,arg,2,3);
      genequal(arg[0],gp->gpname,&var,&binf,&tv);
      seq=arg[1];
      flag=(nb==3)?arg[2]:newsmall(0);
      newdecl((1<<Cauto)|(1<<Cconst),Gvec,-1,&vinf);
      vfvdat=newnode(Fentry,newentry("fv_data"),-1);
      vfvnext=newnode(Fentry,newentry("fv_next"),-1);
      bseq=addseqright(bseq,geninsertvar(binf,newleaf(vinf)));
      genblock(bseq,-1);
      sav0=s_ctx.n;
      bl0=newblock();
      pushvar(vfvdat,(1<<Cuser)|(1<<Cconst),Ggen,-1);
      pushvar(vfvnext,(1<<Cuser)|(1<<Cconst),Gfunc_GG,-1);
      pushvar(var,(1<<Cuser)|(1<<Ccompo),Gvec,newcall("_forvec_start",
            newnode(Flistarg,newnode(Flistarg,newnode(Flistarg,
                  newleaf(vinf),newleaf(flag)),
                newnode(Frefarg,newnode(Fentry,newentry("fv_data"),-1),-1)),
              newnode(Frefarg,newnode(Fentry,newentry("fv_next"),-1),-1))));
      savx=s_ctx.n;
      bl1=newblock();
      bl2=newblock();
      if (autogc)
        gpm=gengerepile(bl2);
      next = newnode1(Fassign,newleaf(var),newnode(Ftag,
            newcall("_call_GG",newnode(Flistarg,newnode(Flistarg,
                  newleaf(vfvnext),newleaf(vfvdat)),newleaf(var))),Gvec));
      lab=makesubblock_label(seq,Lbreak);
      if (lab.ne)
        seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
      if (autogc)
        seq=addseqright(seq,newcall("_gerepilemany",gpm));
      arg[0]=GNOARG;
      arg[1]=newleaf(var);
      arg[3]=next;
      arg[2]=seq;
      aseq=newcall("for",stacktoargs(arg,4));
      if (lab.go)
        aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
      makeblocks3(bl0,bl1,bl2,n,bseq,aseq,ret,sav0,savx);
    }
    break;
  case GPfordivfactored:
  case GPfordiv:
    {
      int vdiv,div;
      int vloop;
      int factored = gp->spec == GPfordivfactored;
      int vt = factored ? Gvec: Gint;
      nb=genlistargs(n,arg,3,3);
      seq=arg[2];
      checkisvar(arg[1],gp->gpname,&var,&tv);
      newdecl(1<<Cconst,Gvec,-1,&vdiv);
      div=newcall("divisors", factored ?
          newnode(Flistarg, arg[0], newsmall(1)) : arg[0]);
      bseq=addseqright(bseq,geninsertvar(div,vdiv));
      genblock(bseq,-1);
      savx=s_ctx.n;
      bl1=newblock();
      bl2=newblock();
      if (autogc)
        gpm=gengerepile(bl2);
      newdecl(0,Gsmall,-1,&vloop);
      if (tv==Gnotype) tv = vt;
      pushvar(var,1<<Cuser,tv,-1);
      lab=makesubblock_label(seq,Lbreak);
      if (lab.ne)
        seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
      if (autogc)
        seq=addseqright(seq,newcall("_gerepilemany",gpm));
      seq=addseqleft(
          newnode1(Fassign,newleaf(var),
            newnode(Ftag,
              newcoeff(newleaf(vdiv),newleaf(vloop),-1), vt)),seq);
      arg[0]=newnode1(Fassign,vloop,newsmall(1));
      arg[1]=newopcall(OPle,newleaf(vloop),newcall("length",vdiv));
      arg[2]=seq;
      arg[3]=newopcall(OPpp,newleaf(vloop),-1);
      aseq=newnode(tree[n].f,newentry("for"),stacktoargs(arg,4));
      if (lab.go)
        aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
      makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
      break;
    }
  case GPforstep:
    {
      int bstep,lstep,sstep;
      int vstep,vindex,vsign;
      int dosign,isvec;
      nb=genlistargs(n,arg,4,4);
      genequal(arg[0],gp->gpname,&var,&binf,&tv);
      bsup=arg[1];bstep=arg[2];seq=arg[3];
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vinf);
      bseq=addseqright(bseq,geninsertvar(binf,newleaf(vinf)));
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup);
      bseq=addseqright(bseq,geninsertvar(bsup,newleaf(vsup)));
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vstep);
      bseq=addseqright(bseq,geninsertvar(bstep,newleaf(vstep)));
      if (guesstype(bstep)==Gvec)
      {
        int sarg[3];
        isvec=1; dosign=1;
        newdecl((1<<Cauto)|(1<<Cconst),Gsmall,-1,&vindex);
        newdecl((1<<Cauto)|(1<<Cconst),Gsmall,-1,&lstep);
        bseq=addseqright(bseq,geninsertvar(newcall("length",newleaf(vstep)),newleaf(lstep)));
        sarg[0] = newnode1(Fassign,newleaf(vindex),newsmall(1));
        sarg[1] = newleaf(lstep);
        sarg[2] = newnode(Fmatcoeff,newleaf(vstep),
            newnode(Fmatrix,newnode(Frange,newleaf(vindex),GNORANGE),-1));
        sstep = newcall("sum", stacktoargs(sarg,3));
        if (tv==Gnotype) tv=Ggen;
      }
      else
      {
        isvec=0;
        dosign=tree[bstep].f!=Fsmall;
        if (dosign)
        {
          sstep = newleaf(vstep);
          if (tv==Gnotype) tv=Ggen;
        } else
          genequaltosmall(binf,bsup,&tv,Ggen);
      }
      if (dosign)
      {
        newdecl((1<<Cauto)|(1<<Cconst),Gbool,-1,&vsign);
        sstep = newopcall(OPg,sstep, newsmall(0));
        bseq=addseqright(bseq,geninsertvar(sstep,newleaf(vsign)));
      }
      genblock(bseq,-1);
      savx=s_ctx.n;
      bl1=newblock();
      bl2=newblock();
      if (autogc)
        gpm=gengerepile(bl2);
      pushvar(var,1<<Cuser,tv,-1);
      lab=makesubblock_label(seq,Lbreak);
      if (lab.ne)
        seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
      if (autogc)
        seq=addseqright(seq,newcall("_gerepilemany",gpm));
      arg[0]=newnode1(Fassign,var,vinf);
      if ( !dosign )
      {
        if ( tree[bstep].x>0 )
          arg[3]=newopcall(OPle,newleaf(var),vsup);
        else
          arg[3]=newopcall(OPge,newleaf(var),vsup);
        nb=4;
      }
      else
      {
        arg[3]=newleaf(vsign);
        arg[4]=newopcall(OPle,newleaf(var),vsup);
        arg[5]=newopcall(OPge,newleaf(var),vsup);
        nb=6;
      }
      if (isvec)
      {
        arg[2]=newopcall(OPpe,var,newcoeff(vstep,vindex,-1));
        arg[6]=vindex;
        arg[7]=newleaf(lstep);
        nb=8;
      }
      else
        arg[2]=newopcall(OPpe,var,vstep);
      arg[1]=seq;
      if (isvec)
        aseq=newcall("_forstepvec",stacktoargs(arg,nb));
      else
        aseq=newnode(tree[n].f,tree[n].x,stacktoargs(arg,nb));
      if (lab.go)
        aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
      makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    }
    break;
  case GPprodeuler:
    nb=genlistargs(n,arg,3,3);
    seq = arg[2];
    newdecl(0,Ggen,-1,&ret);
    savx=s_ctx.n;
    bl1=newblock();
    arg[2]=geninsertvarop(seq,ret,OPme);
    aseq=newnode(Fseq,
            geninsertvar(newnode(Fconst,newsmallrealvalue(1),-1),newleaf(ret)),
                 newcall("forprime",stacktoargs(arg,3)));
    genblock(aseq,-1);
    makeblock(bl1,n,aseq,ret,savx);
    break;
  case GPforprime:
    nb=genlistargs(n,arg,3,3);
    genequal(arg[0],gp->gpname,&var,&binf,&tv);
    bsup=arg[1];seq=arg[2];
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vinf);
    bseq=addseqright(bseq,geninsertvar(binf,newleaf(vinf)));
    if (bsup!=GNOARG || FC_forprime_init<0)
    {
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup);
      bseq=addseqright(bseq,geninsertvar(bsup,newleaf(vsup)));
    }
    genblock(bseq,-1);
    savx=s_ctx.n;
    bl1=newblock();
    bl2=newblock();
    if (autogc)
      gpm=gengerepile(bl2);
    genequaltosmall(binf,bsup,&tv,FC_forprime_init>=0?Gint:Gsmall);
    pushvar(var,1<<Cuser,tv,FC_forprime_init>=0?-1:newsmall(0));
    vprime=newnode(Fentry,newentry("primepointer"),-1);
    pushvar(vprime,1<<Cuser,Gforprime,FC_forprime_init>=0?-1:newcall("_diffptr",-1));
    lab=makesubblock_label(seq,Lbreak);
    if (lab.ne)
      seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
    if (autogc)
      seq=addseqright(seq,newcall("_gerepilemany",gpm));
    if (FC_forprime_init>=0)
    {
      int args[3],bseq;
      int nbs=3;
      args[0]=vprime; args[1]=vinf;
      if (bsup==GNOARG) nbs=2; else args[2]=vsup;
      aseq=addseqleft(newcall((tv==Gsmall?"_u_forprime_init":"_forprime_init"),
            stacktoargs(args,nbs)),aseq);
      bseq=newnode1(Fassign,var,newcall(tv==Gsmall?"_u_forprime_next":"_forprime_next_",vprime));
      tree[bseq].m=Mparens;
      aseq=addseqright(aseq,newcall("_whilenull",newnode(Flistarg,bseq,seq)));
    }
    else
    {
      /*if (vsup>maxprime()) err(primer1);*/
      aseq=addseqleft(newcall("if",newnode(Flistarg,
              newopcall(OPg,vsup,newcall("_maxprime",-1)),
              newcall("_err_primes",-1)))
          ,aseq);
      if (tree[binf].f!=Fsmall || tree[binf].x>2)
        /*if(var<vinf) continue;*/
        seq=addseqleft(newcall("if",newnode(Flistarg,
                newopcall(OPl,newleaf(var),vinf),
                newcall("next",-1)
                )),seq);
      seq=addseqleft(newcall("if",newnode(Flistarg,
              newopcall(OPg,newleaf(var),vsup),
              newcall("break",-1)
              )),seq);
      /*NEXT_PRIME_VIA_DIFF(var,vprime);*/
      seq=addseqleft(newcall("_forprime_next",newnode(Flistarg,
              newleaf(var),newleaf(vprime))),seq);
      aseq=addseqright(aseq,newcall("_doloop",seq));
    }
    if (lab.go)
      aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
    makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    break;
  case GPsum:
  case GPprod:
    nb=genlistargs(n,arg,3,4);
    genequal(arg[0],gp->gpname,&var,&binf,&tv);
    bsup=arg[1];seq=arg[2];
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vinf);
    bseq=addseqright(bseq,geninsertvar(binf,newleaf(vinf)));
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup);
    bseq=addseqright(bseq,geninsertvar(bsup,newleaf(vsup)));
    newdecl(0,Ggen,-1,&ret);
    if (nb==3)
      aseq=addseqright(aseq,geninsertvar(newsmall(gp->spec==GPsum?0:1),
            newleaf(ret)));
    else
      bseq=addseqright(bseq,geninsertvar(arg[3],newleaf(ret)));
    genblock(bseq,-1);
    savx=s_ctx.n;
    bl1=newblock();
    bl2=newblock();
    if (autogc)
      gpm=gengerepile(bl2);
    genequaltosmall(binf,bsup,&tv,Gint);
    pushvar(var,1<<Cuser,tv,-1);
    seq=geninsertvarop(seq,ret,(gp->spec==GPsum?OPpe:OPme));
    makesubblock_label(seq,Lnobrk);
    if (autogc)
      seq=addseqright(seq,newcall("_gerepilemany",gpm));
    arg[0]=newnode1(Fassign,var,vinf);
    arg[1]=newopcall(OPle,newleaf(var),vsup);
    arg[2]=seq;
    arg[3]=newopcall(OPpp,newleaf(var),-1);
    aseq=addseqright(aseq,newcall("for",stacktoargs(arg,4)));
    makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    break;
  case GPsumdiv:
    {
      int vdiv,div;
      int vloop;
      nb=genlistargs(n,arg,3,3);
      seq=arg[2];
      checkisvar(arg[1],gp->gpname,&var,&tv);
      newdecl(1<<Cconst,Gvec,-1,&vdiv);
      div=newcall("divisors",arg[0]);
      bseq=addseqright(bseq,geninsertvar(div,vdiv));
      newdecl(0,Ggen,-1,&ret);
      bseq=addseqright(bseq,geninsertvar(newsmall(0),newleaf(ret)));
      genblock(bseq,-1);
      savx=s_ctx.n;
      bl1=newblock();
      bl2=newblock();
      if (autogc)
        gpm=gengerepile(bl2);
      newdecl(0,Gsmall,-1,&vloop);
      if (tv==Gnotype) tv=Gint;
      pushvar(var,1<<Cuser,tv,-1);
      seq=geninsertvarop(seq,ret,OPpe);
      makesubblock_label(seq,Lnobrk);
      if (autogc)
        seq=addseqright(seq,newcall("_gerepilemany",gpm));
      seq=addseqleft(
          newnode1(Fassign,newleaf(var),
            newnode(Ftag,
              newcoeff(newleaf(vdiv),newleaf(vloop),-1),Gint)),seq);
      arg[0]=newnode1(Fassign,vloop,newsmall(1));
      arg[1]=newopcall(OPle,newleaf(vloop),newcall("length",vdiv));
      arg[2]=seq;
      arg[3]=newopcall(OPpp,newleaf(vloop),-1);
      aseq=newnode(tree[n].f,newentry("for"),stacktoargs(arg,4));
      makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
      break;
    }
  case GPuntil:
  case GPwhile:
    nb=genlistargs(n,arg,2,2);
    bl1=newblock();
    savx=s_ctx.n;
    seq=arg[1];
    if (seq != GNOARG)
    {
      if(autogc)
        gpm=gengerepile(bl1);
      genblock_label(arg[0],n,Lnobrk);
      lab=makesubblock_label(seq,Lbreak);
      if (lab.ne)
        seq=newseq(seq,newcall("_label",newsmall(lab.ne)));
      if (autogc)
        seq=addseqright(seq,newcall("_gerepilemany",gpm));
    }
    else
      genblock(arg[0],n);
    arg[1]=seq;
    aseq=newnode(tree[n].f,tree[n].x,stacktoargs(arg,nb));
    if (seq != GNOARG && lab.go)
      aseq=newseq(aseq,newcall("_label",newsmall(lab.go)));
    makeblock(bl1,n,aseq,ret,savx);
    break;
  case GPvector:
  case GPvectorv:
  case GPvectorsmall:
    nb=genlistargs(n,arg,1,3);
    bsup=arg[0];seq=nb<3?newsmall(0):arg[2];
    newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup);
    bseq=addseqright(bseq,geninsertvar(bsup,newleaf(vsup)));
    genblock(bseq,-1);
    newdecl(0,gp->spec==GPvectorsmall?Gvecsmall:Gvec,-1,&ret);
    savx=s_ctx.n;
    bl1=newblock();
    bl2=newblock();
    if (nb>=2)
    {
      checkisvar(arg[1],gp->gpname,&var,&tv);
      pushvar(var,1<<Cuser,tv==Gnotype?Gsmall:tv,-1);
    }
    else
      newdecl(0,Gsmall,-1,&var);
    aseq=addseqright(aseq,newgetg(ret,newleaf(vsup),
          gp->spec==GPvector?"t_VEC":gp->spec==GPvectorv?"t_COL":"t_VECSMALL"));
    arg[0]=newnode1(Fassign,var,newsmall(1));
    arg[1]=newopcall(OPle,newleaf(var),vsup);
    arg[2]=newcoeff(newleaf(ret),newleaf(var),-1);
    arg[2]=geninsertvar(seq,arg[2]);
    makesubblock_label(arg[2],Lnobrk);
    arg[3]=newopcall(OPpp,newleaf(var),-1);
    aseq=addseqright(aseq,newcall("for",stacktoargs(arg,4)));
    makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    break;
  case GPmatrix:
    {
      int var1,bsup1,vsup1,var2,bsup2,vsup2;
      int tv1,tv2;
      nb=genlistargs(n,arg,2,5);
      bsup2=arg[0];bsup1=arg[1];
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup1);
      newdecl((1<<Cauto)|(1<<Cconst),Gempty,-1,&vsup2);
      bseq=addseqright(bseq,geninsertvar(bsup1,newleaf(vsup1)));
      bseq=addseqright(bseq,geninsertvar(bsup2,newleaf(vsup2)));
      genblock(bseq,-1);
      newdecl(0,Gvec,-1,&ret);
      savx=s_ctx.n;
      bl1=newblock();
      bl2=newblock();
      if (nb>=3)
      {
        checkisvar(arg[2],gp->gpname,&var2,&tv2);
        pushvar(var2,1<<Cuser,tv2==Gnotype?Gsmall:tv2,-1);
      }
      else
        newdecl(0,Gsmall,-1,&var2);
      if (nb>=4)
      {
        checkisvar(arg[3],gp->gpname,&var1,&tv1);
        pushvar(var1,1<<Cuser,tv1==Gnotype?Gsmall:tv1,-1);
      }
      else
        newdecl(0,Gsmall,-1,&var1);
      if(nb<5)
        seq=newsmall(0);
      else
        seq=arg[4];
      aseq=newgetg(newcoeff(newleaf(ret),newleaf(var1),-1),
          newleaf(vsup2),"t_COL");
      arg[0]=newnode1(Fassign,var2,newsmall(1));
      arg[1]=newopcall(OPle,newleaf(var2),vsup2);
      arg[3]=newopcall(OPpp,newleaf(var2),-1);
      arg[2]=newcoeff(newleaf(ret),newleaf(var2),newleaf(var1));
      arg[2]=geninsertvar(seq,arg[2]);
      makesubblock_label(arg[2],Lnobrk);
      arg[2]=addseqright(aseq,newcall("for",stacktoargs(arg,4)));
      aseq=newgetg(ret,newleaf(vsup1),"t_MAT");
      arg[0]=newnode1(Fassign,var1,newsmall(1));
      arg[1]=newopcall(OPle,newleaf(var1),vsup1);
      arg[3]=newopcall(OPpp,newleaf(var1),-1);
      aseq=addseqright(aseq,newcall("for",stacktoargs(arg,4)));
      makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
    }
    break;
  case GPglobal:
  case GPlocal:
    {
      int aseq=GNIL;
      int flag=1<<Cuser;
      int mint=(tree[p].f==Ftag)?tree[p].y:-1;
      if (gp->spec==GPglobal)
        flag|=(1<<Cglobal)|(1<<Cfunction);
      savx=s_ctx.n;
      (void)genblockdeclaration(y,n,flag,mint,&aseq);
      makeblock(newblock(),n,aseq,-1,s_ctx.n);
      if (tree[p].f==Ftag) tree[p]=tree[n];
    }
    break;
  case GPself:
    {
      int str = newstringnode(lfunc[currfunc].gpname,-1);
      int aseq = newcall("_strtoclosure",str);
      tree[n] = tree[aseq];
    }
    break;
  case GPlocalprec:
    {
      char buf[64];
      int vn;
      sprintf(buf,"prec%d",++preclevel);
      vn = newnode(Fentry, newentry(strdup(buf)), -1);
      pushvar(vn, (1<<Cauto), Gsmall, -1);
      y = newnode1(Fassign, vn, newcall("_ndec2prec", y));
      genblock(y, n);
      tree[n].y = y;
    }
    break;
  case GPlocalbitprec:
    {
      char buf[64];
      int vn;
      sprintf(buf,"bitprec%d",++preclevel);
      vn = newnode(Fentry, newentry(strdup(buf)), -1);
      pushvar(vn, (1<<Cauto), Gsmall, -1);
      y = newnode1(Fassign, vn, newnode(Ftag, newleaf(y), Gsmall));
      genblock(y, n);
      tree[n].y = y;
    }
    break;
  case GPreturn:
    {
      int iferr_reset=0;
      if (y==GNOARG || y<0)
        y=tree[n].y=GNIL;
      else
        genblock(y,n);
      iferr_reset=check_labels(s_label.n,Liferr)==1;
      /* If we don't do garbage collecting, don't bother adding blocks.*/
      if (!autogc && !iferr_reset) break;
      if (tree[y].f==Fentry && !(ctxstack[getvarerr(y)].flag&(1<<Cglobal)))
      {
        var=y;
        bseq=-1;
        y=newleaf(var);
      }
      else if (tree[y].f==Fblock)
      {
        /*Use the block value variable as the return variable*/
        var=block[tree[y].x].ret;
        bseq=y;
        y=newleaf(var);
      }
      else if (tree[y].f!=Fsmall)
      {
        /*Use the returned variable as the return variable*/
        newdecl((1<<Cauto),Gempty,-1,&var);
        bseq=geninsertvar(y,var);
        y=newleaf(var);
      }
      bl1=newblock();
      bl2=newblock();
      savx=s_ctx.n;
      block[bl2].gc|=(1<<GCneeded)|(1<<GCglobal)|(1<<GCreturn);
      block[bl2].egc=var;
      if (autogc)
      {
        gpm=gengerepilereturn(bl2);
        aseq=addseqright(aseq,newcall("_gerepilemany",gpm));
      }
      if (iferr_reset)
        aseq=addseqright(aseq,newcall("_iferr_CATCH_reset",newsmall(0)));
      aseq=addseqright(aseq,newnode(tree[n].f,tree[n].x,y));
      makeblocks(bl1,bl2,n,bseq,aseq,ret,savx);
      break;
    }
  case GPbreak:
  case GPnext:
    {
      static int labbrk=0;
      if (y>=0 && y!=GNOARG)
      {
        if (tree[y].f!=Fsmall || tree[y].x<=0)
          die(n,"%s(n) only supported for constant n>0",gp->gpname);
        nb=tree[y].x;
      }
      else
      {
        nb=1;
        tree[n].y=GNOARG;
      }
      if (check_labels(nb,Lnobrk))
        die(n,"%s not allowed here",gp->gpname);
      if (nb>1)
      {
        int l = s_label.n-nb;
        if (gp->spec==GPbreak)
        {
          if (label[l].go==0)
            label[l].go=++labbrk;
          tree[y].x=labbrk;
        }
        else
        {
          if (label[l].ne==0)
            label[l].ne=++labbrk;
          tree[y].x=labbrk;
        }
      }
      if (check_labels(nb,Liferr))
      {
        int ie = newsmall(get_iferr(nb));
        aseq=addseqleft(newcall("_iferr_CATCH_reset",ie),newleaf(n));
        makeblock(newblock(),n,aseq,-1,s_ctx.n);
      }
    }
    break;
  case GPiferr:
    {
      Clabel_t lab;
      nb=genlistargs(n,arg,3,4);
      checkisvar(arg[1],gp->gpname,&var,&tv);
      seq=arg[0];
      bseq=arg[2];
      if (tv==Gnotype)
      {
        tv=Gerror;
        if (tv<0) die(n,"iferr not supported by your GP version");
      }
      if (p>=0 && tree[p].f!=Fseq)
        newdecl((1<<Cauto)|(1<<Cimmutable),Gvoid,GNIL,&ret);
      seq=geninsertvar(seq,ret);
      bseq=geninsertvar(bseq,ret);
      lab=makesubblock_label(seq,Liferr);
      savx=s_ctx.n;
      pushvar(var,1<<Cuser,tv,newcall("_iferr_error",-1));
      if(nb==4)
      {
        int pred = newcall("if", newnode(Flistarg,
                      newnode(Flistarg,arg[3],GNOARG),
                      newcall("_iferr_rethrow",var)));
        bseq = addseqleft(pred,bseq);
      }
      makesubblock_ctx(bseq,savx);
      arg[0]=newcall("_iferr_CATCH", newsmall(lab.ie));
      arg[1]=bseq;
      arg[2]=seq;
      arg[3]=newcall("_iferr_ENDCATCH",newsmall(lab.ie));
      aseq=addseqright(aseq,newcall("iferr",stacktoargs(arg,4)));
      makeblock(newblock(),n,aseq,ret,s_ctx.n);
      break;
    }
  default:
    /*treat as a normal function*/
    genblock(tree[n].y,n);
  }
}
void gentypemode(int nb, int *arg, int *mode)
{
  int i;
  *mode=0;
  for (i=0;i<nb;i++)
    if (arg[i]!=GNOARG)
    {
      gentype(arg[i]);
      *mode|=(tree[arg[i]].m&MODHERIT);
    }
}

int gentypefuncspec(int n, gpfunc *gp)
{
  int arg[STACKSZ];
  int nb;
  int y=tree[n].y;
  switch(gp->spec)
  {
  case GPif:
    nb=genlistargs(n,arg,2,3);
    gentype(arg[0]);
    gentype(arg[1]);
    tree[n].m=(1<<Melse)|((tree[arg[0]].m|(tree[arg[1]].m&~(1<<Mterm)))&MODHERIT);
    if (nb==3)
    {
      gentype(arg[2]);
      /*We must not set Mterm if we are not sure.
        In this case we must set Msidef instead.*/
      tree[n].m|=((tree[arg[2]].m&(~(1<<Mterm)))&MODHERIT);
      if ((tree[arg[1]].m&(1<<Mterm)) && (tree[arg[2]].m&(1<<Mterm)))
        tree[n].m|=(1<<Mterm);
      else
        if ((tree[arg[1]].m&(1<<Mterm)) || (tree[arg[2]].m&(1<<Mterm)))
          tree[n].m|=(1<<Msidef);
      return typemax[tree[arg[1]].t][tree[arg[2]].t];
    }
    if ((tree[arg[1]].m&(1<<Mterm)))
      tree[n].m|=(1<<Msidef);
    return tree[arg[1]].t;
  case GPwhilenull:
    nb=genlistargs(n,arg,2,2);
    gentypemode(nb,arg,&tree[n].m);
    tree[n].m&=~(1<<Mterm);
    return Gvoid;
  case GPiferr:
    nb=genlistargs(n,arg,4,4);
    gentypemode(4,arg,&tree[n].m);
    tree[arg[1]].m|=1<<Muntil;
    tree[arg[2]].m|=1<<Muntil;
    tree[arg[3]].m|=1<<Muntil;
    tree[n].m&=~(1<<Mterm);
    return typemax[tree[arg[1]].t][tree[arg[2]].t];
  case GPdoloop:
    gentype(y);
    tree[n].m=tree[y].m&MODHERIT;
    return Gvoid;
  case GPfor:
    nb=genlistargs(n,arg,4,4);
    gentypemode(nb,arg,&tree[n].m);
    tree[n].m&=~(1<<Mterm);
    return Gvoid;
  case GPforstep:
  case GPforstepvec:
    nb=genlistargs(n,arg,4,8);
    gentypemode(nb,arg,&tree[n].m);
    tree[n].m&=~(1<<Mterm);
    return Gvoid;
  case GPuntil:
  case GPwhile:
    nb=genlistargs(n,arg,2,2);
    gentype(arg[0]);
    gentype(arg[1]);
    if(gp->spec==GPwhile)
      tree[n].m=(tree[arg[0]].m|(tree[arg[1]].m&~(1<<Mterm)))&MODHERIT;
    else
    {
      int a = arg[1];
      tree[n].m=((tree[arg[0]].m|tree[arg[1]].m)&MODHERIT);
      tree[a].m|=(1<<Muntil);
    }
    return Gvoid;
  case GPreturn:
    {
      gpfunc *cf=lfunc+currfunc;
      int t;
      if (y>=0)
      {
        gentype(y);
        tree[n].m=(1<<Mterm)|(tree[y].m&MODHERIT);
        t=typemax[functype(*cf)][tree[y].t];
      }
      else
      {
        tree[n].m=(1<<Mterm);
        t=typemax[functype(*cf)][Gvoid];
      }
      if (!is_subtype(t,functype(*cf)))
      {
        if(debug==2)
          fprintf(stderr,"%s returned %s now %s (%s)\n",cf->gpname,GPname(functype(*cf)),GPname(t),GPname(tree[y].t));
        functype(*cf)=t;
        lastpass++;
      }
    }
    return Gempty;
  case GPinstall:
  case GPaddhelp:
    tree[n].m=0;
    return Gvoid;
  case GPmakevec:
    gentype(y);
    tree[n].m=(tree[y].m&MODHERIT)|(1<<Msidef);
    return Gvec;
  case GPmakevecsmall:
    gentype(y);
    tree[n].m=(tree[y].m&MODHERIT)|(1<<Msidef);
    return Gvecsmall;
  case GPclosure:
    nb=genlistargs(n,arg,1,STACKSZ);
    gentypemode(nb,arg,&tree[n].m);
    if ((tree[n].m&(1<<Mprec))==0)
    {
      int nf = findfunction(entryname(arg[0]));
      if (nf>=0 && lfunc[nf].spec==GPuser && lfunc[nf].user->wrapper>=0 && (funcmode(lfunc[nf])&(1<<Mprec)))
        tree[n].m|=(1<<Mprec);
    }
    return Gclosure;
  default:
    if (gp->spec>0)
      die(n,"gentypefuncspec : func spec not implemented");
    else
      die(n,"Internal error : gentypefuncspec called with no spec");
  }
}

void gencopyarg(FILE *fout)
{
  gpfunc *gp=lfunc+currfunc;
  userfunc *ufunc=gp->user;
  context *fc=block+ufunc->bl;
  int n=tree[ufunc->defnode].y;
  long i;
  if (tree[n].x!=ufunc->bl || !fc->var) return;
  for (i=0;i<fc->s.n;i++)
    if (fc->c[i].flag&(1<<Carg))
    {
      ctxvar *v=fc->c+i;
      long j;
      for(j=0;j<fc->v.n;j++)
        if (fc->var[j].f==AFassigncompo && fc->var[j].idx==i+fc->savb-fc->s.n )
        {
          genindent(fout);
          fprintf(fout,"%s = ",v->cvar);
          genfuncbydesc1(fout,v->node,FC_copy,n);
          gensemicolon(fout,v->node);
          break;
        }
    }
}

void geninitfunc(FILE *fout)
{
  gpfunc *gp=lfunc+currfunc;
  int i;
  for (i=0;i<gp->user->savb;i++)
  {
    ctxvar *v=ctxstack+i;
    if (!(v->flag&(1<<Cundeclared)))
      continue;
    /* We need the varlist which is not available at gentype() time, so we do
     * it here instead. Given that v->initval is equal to 'v, this is harmless.
     */
    gentype(v->initval);
    genindent(fout);
    fprintf(fout,"%s = ", v->cvar);
    gencast(fout,v->initval,vartype(*v));
    fprintf(fout,";\n");
  }
}

void genentryspec(FILE *fout, int n, gpfunc *gp)
{
  int arg[STACKSZ];
  int nb;
  int y=tree[n].y;
  int i;
  context *bl;
  switch(gp->spec)
  {
    case GPbreak:
    case GPnext:
      if (y>=0 && tree[y].f==Fsmall)
        fprintf(fout,"goto label%d",tree[y].x);
      else
        fprintf(fout,gp->spec==GPbreak?"break":"continue");
      break;
    case GPif:
      nb=genlistargs(n,arg,2,3);
      genindentline(fout,"if (");
      gencast(fout,arg[0],Gbool);
      fprintf(fout,")\n");
      if (nb==2)
        genbrace(fout,arg[1]);
      else
      {
        genbrace(fout,arg[1]);
        genindentline(fout,"else\n");
        genbrace(fout,arg[2]);
      }
      break;
    case GPiferr:
      nb=genlistargs(n,arg,4,4);
      genindent(fout);
      gencode(fout,arg[0]);
      fprintf(fout,"\n");
      genbrace(fout,arg[1]);
      fprintf(fout," pari_TRY\n");
      genbrace(fout,arg[2]);
      fprintf(fout," ");
      gencode(fout,arg[3]);
      fprintf(fout,"\n");
      break;
    case GPwhilenull:
      nb=genlistargs(n,arg,2,2);
      genindentline(fout,"while (");
      genparens(fout,arg[0]);
      fprintf(fout,")\n");
      genbrace(fout,arg[1]);
      break;
    case GPwhile:
      nb=genlistargs(n,arg,2,2);
      if (tree[arg[0]].f!=Fseq && tree[arg[0]].f!=Fblock)
      {
        genindentline(fout,"while (");
        gencast(fout,arg[0],Gbool);
        if (arg[1]==GNOARG)
          fprintf(fout,");\n");
        else
        {
          fprintf(fout,")\n");
          genbrace(fout,arg[1]);
        }
      }
      else
      {
        int x=tree[arg[0]].x;
        int y=tree[arg[0]].y;
        if (tree[arg[0]].f==Fblock)
        {
          y=block[x].ret;
          x=tree[arg[0]].y;
        }
        genindentline(fout,"for(;;)\n");
        genindentline(fout,"{\n");
        indent++;
        genindentseq(fout,x);
        gencode(fout,x);
        gensemicolon(fout,x);
        if (tree[y].f==Fblock)
        {
          gencode(fout,y);
          y=block[tree[y].x].ret;
        }
        genindentline(fout,"if (");
        gencast(fout,y,Gnegbool);
        fprintf(fout,")\n");
        indent++;
        genindentline(fout,"break;\n");
        indent--;
        genindentseq(fout,arg[1]);
        gencode(fout,arg[1]);
        gensemicolon(fout,arg[1]);
        indent--;
        genendbrace(fout,n);
      }
      break;
    case GPuntil:
      nb=genlistargs(n,arg,2,2);
      if ( tree[arg[0]].f!=Fseq && tree[arg[0]].f!=Fblock)
      {
        if (arg[1]!=GNOARG)
        {
          genindentline(fout,"do\n");
          genbrace(fout,arg[1]);
          fprintf(fout," while(");/*KB style*/
        }
        else
          genindentline(fout,"while(");
        gencast(fout,arg[0],Gnegbool);
        fprintf(fout,");\n");
      }
      else
      {
        int x=tree[arg[0]].x;
        int y=tree[arg[0]].y;
        if (tree[arg[0]].f==Fblock)
        {
          y=block[x].ret;
          x=tree[arg[0]].y;
        }
        genindentline(fout,"for(;;)\n");
        genindentline(fout,"{\n");
        indent++;
        genindentseq(fout,arg[1]);
        gencode(fout,arg[1]);
        gensemicolon(fout,arg[1]);
        genindentseq(fout,x);
        gencode(fout,x);
        gensemicolon(fout,x);
        if (tree[y].f==Fblock)
        {
          genindent(fout);
          gencode(fout,y);
          y=block[tree[y].x].ret;
        }
        genindentline(fout,"if (");
        gencast(fout,y,Gbool);
        fprintf(fout,")\n");
        indent++;
        genindentline(fout,"break;\n");
        indent--;
        indent--;
        genendbrace(fout,n);
      }
      break;
    case GPfor:
      nb=genlistargs(n,arg,4,4);
      genindentline(fout,"for (");
      if (arg[0]!=GNOARG)
        gencode(fout,arg[0]);
      else
        fprintf(fout,"  ");
      fprintf(fout,"; ");
      if (arg[1]!=GNOARG)
        gencode(fout,arg[1]);
      fprintf(fout,"; ");
      if (arg[3]!=GNOARG)
        gencode(fout,arg[3]);
      fprintf(fout,")\n");
      genbrace(fout,arg[2]);
      break;
    case GPdoloop:
      genindentline(fout,"for (;;)\n");
      genbrace(fout,y);
      break;
    case GPforstep:
      nb=genlistargs(n,arg,4,6);
      genindentline(fout,"for (");
      gencode(fout,arg[0]);
      fprintf(fout,"; ");
      if ( nb==4 )
        gencode(fout,arg[3]);
      else
      {
        gencode(fout,arg[3]);
        fprintf(fout,"?");
        gencode(fout,arg[4]);
        fprintf(fout,":");
        gencode(fout,arg[5]);
      }
      fprintf(fout,"; ");
      gencode(fout,arg[2]);
      fprintf(fout,")\n");
      genbrace(fout,arg[1]);
      break;
#if 0
      forstep(x=binf,bsup,step,SEQ)
      {
        int idx;

      for( idx=0,x=binf; x<=bsup; i=i==lg(step)?1:i+1,x+=step[i])
      {
      }
    }
#endif
  case GPforstepvec:
    nb=genlistargs(n,arg,8,8);
    genindentline(fout,"for (");
    gencode(fout,arg[6]);
    fprintf(fout," = 0, ");
    gencode(fout,arg[0]);
    fprintf(fout,"; ");
    gencode(fout,arg[3]);
    fprintf(fout,"?");
    gencode(fout,arg[4]);
    fprintf(fout,":");
    gencode(fout,arg[5]);
    fprintf(fout,"; ");
    gencode(fout,arg[6]);
    fprintf(fout," = ");
    gencode(fout,arg[6]);
    fprintf(fout," == ");
    gencast(fout,arg[7],Gsmall);
    fprintf(fout," ? 1 :");
    gencode(fout,arg[6]);
    fprintf(fout," + 1, ");
    gencode(fout,arg[2]);
    fprintf(fout,")\n");
    genbrace(fout,arg[1]);
    break;
  case GPmakevecsmall:
  case GPmakevec:
    nb=genlistargs(n,arg,1,STACKSZ);
    genindent(fout);
    gencode(fout,arg[0]);
    gensemicolon(fout,arg[0]);
    for(i=1;i<nb;i++)
    {
      genindentseq(fout,arg[i]);
      gencode(fout,arg[i]);
      gensemicolon(fout,arg[i]);
    }
    break;
  case GPgpmany:
    nb=genlistargs(n,arg,2,3);
    bl=block+tree[arg[1]].x;
    if (!(bl->gc&(1<<GCneeded)))
      break;
    genindent(fout);
    if (!bl->g.n)
    {
      genfuncbydesc(fout,0,NULL,FC_avma,n);
      fprintf(fout," = ");
      gencode(fout,arg[0]);
      fprintf(fout,";\n");
    }
    else if (bl->gc&(1<<GCupto))
    {
      int var=bl->gcvar[0];
      int parg[2];
      gencode(fout,var);
      fprintf(fout," = ");
      parg[0]=arg[0]; parg[1]=var;
      genfuncbydesc(fout,2,parg,FC_gerepileupto,n);
      fprintf(fout,";\n");
    }
    else
    {
      int parg[STACKSZ+1];
      /*FIXME:Putting an if statement in the wilderness is not secure*/
      if (nb==3)
      {
        fprintf(fout,"if (");
        if (arg[2]==GNOARG)
        {
          parg[0]=arg[0];
          genfuncbydesc(fout,1,parg,FC_gc_needed,n);
        }
        else
        {
          parg[0]=arg[2]; parg[1]=arg[0];
          genfuncbydesc(fout,2,parg,FC_low_stack_lim,n);
        }
        fprintf(fout,")\n");
        indent++;
        genindent(fout);
        indent--;
      }
      parg[0]=arg[0];
      for(i=0;i<bl->g.n;i++)
        parg[i+1]=bl->gcvar[i];
      genfuncbydesc(fout,bl->g.n+1,parg,FC_gerepileall,n);
      fprintf(fout,";\n");
    }
    break;
  case GPlocalprec:
  case GPlocalbitprec:
    gencode(fout, y);
    preclevel++;
    break;
  case GPreturn:
    fprintf(fout,"return");
    if (functype(lfunc[currfunc])!=Gvoid)
    {
      /*I don't like to see `return ;'*/
      fprintf(fout," ");
      gencast(fout,y,functype(lfunc[currfunc]));
    }
    break;
  case GPlabel:
    fprintf(fout,"label%d:",tree[y].x);
    break;
  case GPcopyarg:
    gencopyarg(fout);
    break;
  case GPinitfunc:
    geninitfunc(fout);
    break;
  case GPaddhelp:
  case GPinstall:
    break;
  case GPclosure:
    nb=genlistargs(n,arg,1,STACKSZ);
    if (FC_strtoclosure >=0)
      genfuncbydesc(fout,nb,arg,FC_strtoclosure,n);
    else
      die(n,"closure not available in this version");
    break;
  default:
    if (gp->spec>0)
      die(n,"genentryspec: func spec not implemented");
    else
      die(n,"Internal error: genentryspec called with no spec");
  }
}
